module net.BurtonRadons.dig.platform.fileSelector;


//private import std.string;

/** The file selector opens up an OS-specific file selection dialog for saving
  * or loading a file or files.  For example, this puts up a file selector for
  * a set of image files:
  *
  * @code
  * char [] [] selectImages ()
  * {
  *     with (new FileSelector (false))
  *     {
  *         multiSelect = true;
  *         addFilter ("All Files", "*");
  *         addFilter ("Image Files", "*.png;*.jpg;*.jpeg");
  *         return run ();
  *     }
  * }
  * @endcode
  */

class FileSelector
{
	private import net.BurtonRadons.dig.platform.base;
    private import net.BurtonRadons.dig.platform.control;
    private import net.BurtonRadons.dig.platform.windows;

    /** An individual filter registered with addFilter. */
    struct Filter
    {
        char [] desc; /**< The descriptive name of the filter. */
        char [] pattern; /**< The semi-colon-separated wildcard pattern, case-insensitive. */
    };

    Control parent; /**< The control spawning this selector or null if none; this has no effect to my knowledge. */
    Filter [] filters; /**< The list of filters registered with addFilter. */
    int defaultFilter; /**< The default filter index, 0 for the custom filter, or an index into filters plus one. */
    Filter custom; /**< The custom filter that holds what filter the user enters into the selection editor. */
    char [] title; /**< The text to use in the title bar of the selector, or null for some default. */
    char [] filename; /**< The default filename to insert in the edit box. */

    bit multiSelect = false; /**< Whether this is a multiple-file selector */
    bit createPrompt = false; /**< If the user selects a file that doesn't exist, whether to give an "Are you sure?" message */
    bit fileMustExist = true; /**< Display a warning if the user selects a nonexistent file */
    bit showReadOnly = false; /**< Show a read-only checkbox */
    bit overwritePrompt = false; /**< If this is a save dialog, ask whether the user wants to overwrite this */
    bit pathMustExist = true; /**< Display a warning if the user selects a nonexistent path */
    bit save = false; /**< Save mode if true, open mode if false */

    /** Set save mode. */
    this (bit save) { this.save = save; }

    /** Set the parent and save mode. */
    this (Control parent, bit save) { this.parent = parent; this.save = save; }

    /** Call addFilter (desc, pattern, false). */
    void addFilter (char [] desc, char [] pattern) { addFilter (desc, pattern, false); }

    /** Add a file filter.
      * The desc is some description (such as "Text Strings")
      * while the pattern is a set of semicolon-separated wildcard
      * filters (such as "*.txt;*.doc;*.bak") that are matched
      * case-independent.  If isdefault is true, this is made the
      * default filter on startup.
      */

    void addFilter (char [] desc, char [] pattern, bit isdefault)
    {
        Filter add;

        add.desc = desc;
        add.pattern = pattern;
        if (isdefault)
            defaultFilter = filters.length + 1;
        filters ~= add;
    }

    /** Call customFilter (desc, null, false) */
    void customFilter (char [] desc) { customFilter (desc, null, false); }

    /** Call customFilter (desc, pattern, false) */
    void customFilter (char [] desc, char [] pattern) { customFilter (desc, pattern, false); }

    /** Set the custom filter.  This is the filter the user writes to
      * when he inputs some wildcard in the selection editor.  @see addFilter.
      */

    void customFilter (char [] desc, char [] pattern, bit isdefault)
    {
        custom.desc = desc;
        if (pattern)
            custom.pattern = pattern.dup;
        else
            custom.pattern = null;

        if (isdefault)
            defaultFilter = 0;
    }

    /** Open up the file selector and start selecting.  If the user hit
      * cancel or some benign error occured, this returns a null array.
      * Otherwise this returns a list of selected files with full paths.
      */

    char [] [] run ()
    {
        OPENFILENAME o;
        char [] filter;
        char [] filename;
        char [] [] result;

        {
            if (filters.length)
            {
                int length = 1;
    
                for (int c; c < filters.length; c ++)
                    length += filters [c].desc.length + filters [c].pattern.length + 2 + 3 + filters [c].pattern.length;

                filter.length = length;
                length = 0;
                for (int c; c < filters.length; c ++)
                {
                    Filter *f = &filters [c];
                    char [] desc;

                    desc = f.desc ~ " (" ~ f.pattern ~ ")";

                    filter [length .. length + desc.length] = desc;
                    filter [length + desc.length] = '\0';
                    length += desc.length + 1;
        
                    filter [length .. length + f.pattern.length] = f.pattern;
                    filter [length + f.pattern.length] = '\0';
                    length += f.pattern.length + 1;
                }

                filter [length] = '\0';
            }

            filename = new char [65535];
            filename [0 .. this.filename.length] = this.filename;

            o.lStructSize = OPENFILENAME.size;
            o.hwndOwner = parent ? parent.digPlatformHWND : (_HANDLE) 0;
            o.hInstance = hinst;
            o.lpstrFilter = (_LPCTSTR) (filters.length ? filter : null);
            o.lpstrCustomFilter = null;
            o.nMaxCustFilter = 0;
            o.nFilterIndex = defaultFilter;
            o.lpstrFile = filename;
            o.nMaxFile = filename.length;
            o.lpstrFileTitle = null;
            o.nMaxFileTitle = 0;
            o.lpstrInitialDir = null;
            o.lpstrTitle = title ? std.string.toStringz (title) : null;
            o.Flags = 0;
            if (multiSelect)
                o.Flags |= OFN_ALLOWMULTISELECT;
            if (createPrompt)
                o.Flags |= OFN_CREATEPROMPT;
            if (fileMustExist)
                o.Flags |= OFN_FILEMUSTEXIST;
            if (!showReadOnly)
                o.Flags |= OFN_HIDEREADONLY;
            if (overwritePrompt)
                o.Flags |= OFN_OVERWRITEPROMPT;
            if (pathMustExist)
                o.Flags |= OFN_PATHMUSTEXIST;
            o.Flags |= OFN_EXPLORER; /* Use modern interface */
            o.Flags |= OFN_LONGNAMES; /* Use long filenames */
            //o.Flags |= OFN_ENABLEHOOK; /* Enable hook function */
            // OFN_ENABLEINCLUDENOTIFY - Enable include notify - brain-drainingly underpowered
            //o.Flags |= OFN_SHOWHELP; /* Display the Help button */
            // OFN_ADDTORECENT
            // OFN_ENABLEHOOK - enable lpfnHook
            // OFN_ENABLEINCLUDENOTIFY
            // OFN_ENABLETEMPLATE
            // OFN_ENABLETEMPLATEHANDLE
            // OFN_EXTENSIONDIFFERENT
            // OFN_NOCHANGEDIR - don't change file directory
            // OFN_NODEREFERENCELINKS - get .lnk files instead of what they point at
            // OFN_NOLONGNAMES - 8.3 names only (ignored with OFN_EXPLORER)
            // OFN_NONETWORKBUTTON - hide the Network button
            // OFN_NOREADONLYRETURN - file doesn't have the Read Only check on and isn't in a write-protected directory
            // OFN_NOTESTFILECREATE - don't make sure that the file can be created
            // OFN_NOVALIDATE - allow invalid characters in the filename
            // OFN_READONLY - Read Only checkbox is automatically selected
            // OFN_SHAREAWARE - ignore network sharing violations
            o.nFileOffset = 0;
            o.nFileExtension = 0;
            o.lpstrDefExt = null;
            o.lCustData = 0; /* Custom data sent to the lpfnHook */
            o.lpfnHook = null;
            o.lpTemplateName = null;

            if ((!save && !GetOpenFileNameA (&o))
              || (save && !GetSaveFileNameA (&o)))
            {
                digPlatformCommDlgExtendedError ();
                return null;
            }

            if (multiSelect && filename [std.string.strlen (filename) + 1])
            {
                char *p;
                int count, blen, len;

                blen = std.string.strlen (filename);
                p = (char *) filename + blen + 1;
                while (*p)
                    p += std.string.strlen (p) + 1, count += 1;

                result.length = count;
                p = (char *) filename + blen + 1;
                for (int c; c < count; c ++, p += len + 1)
                {
                    len = std.string.strlen (p);
                    result [c].length = blen + len + 1;
                    result [c] [0 .. blen] = filename [0 .. blen];
                    result [c] [blen] = '\\';
                    result [c] [blen + 1 .. blen + len + 1] = p [0 .. len];
                }
                
                return result;
            }
            else
            {
                result.length = 1;
                result [0] = filename [0 .. std.string.strlen (filename)].dup;
                return result;
            }

            return null;
        }
    }
    
    /** Open the file selector and start selecting.  Returns null if it was
      * aborted, or the name of the first file if it was successful.
      */
      
    char [] runSingle ()
    {
        char [] [] result = run ();
        
        if (result.length)
            return result [0];
        return null;
    }
}
